﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Web.Mvc;
using BMS.Web.Models;
using BMS.Facade;
using InfoWorld.EVS.CTSMAPI;
using InfoWorld.HL7.ITS;
using BMS.Utils;
using BMS.Web.App_GlobalResource;
using System.Text.RegularExpressions;
using BMS.Facade.Data;

namespace BMS.Web.Controllers
{
    [ValidateInput(false)]
    public class AdminUnavailableReasonController : BaseController
    {
        /// <summary>
        /// Default action for the unavailable reason controller.
        /// </summary>
        /// <returns>The Index view.</returns>
        [ReadPermissionAuthFilterAttribute]
        public ActionResult Index()
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                UnavailableReasonViewModel viewModel = new UnavailableReasonViewModel();                
                FillViewModel(viewModel);
                return View(viewModel);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Post action for the Unavailable Reason controller - used for the Add button.
        /// </summary>
        /// <param name="input">The input(unavailable reason view model).</param>
        /// <returns>The Index view.</returns>
        [HttpPost]
        [UpdatePermissionAuthFilterAttribute]
        public ActionResult Index(UnavailableReasonViewModel input)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                FillViewModel(input);
                UnavailableReasonItemViewModel itemViewModel = new UnavailableReasonItemViewModel();
                if (!string.IsNullOrWhiteSpace(input.UnavailableReasonText))
                {
                    string pattern = "[\\~#%&*{}/:<>?|\"-'^()]";
                    string replacement = " ";
                    Regex regEx = new Regex(pattern);
                    string code = Regex.Replace(regEx.Replace(input.UnavailableReasonText, replacement), @"\s+", " ");
                    code = Constants.NATIONAL_VOCABULARY_CODE_PREFIX + code.Trim().Replace(" ", "_");
                    List<CDWithProperties> reasons = FacadeManager.VocabularyInterface.GetAllUnavailableReasons();
                    if (reasons.Where(a => a.code.Equals(code, StringComparison.InvariantCultureIgnoreCase) || a.displayName.Equals(input.UnavailableReasonText, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault() == null)
                    {
                        itemViewModel.Model = new InfoWorld.EVS.CTSMAPI.CDWithProperties(code.ToUpper(), Constants.UnavailableReasonCodeSystem, Constants.UnavailableReason, "", input.UnavailableReasonText.ToUpper(), null, null, null, new List<ConceptProperty>());
                        ConceptProperty property = new ConceptProperty(new ST("Type"), new ST(input.UnavailableReasonType), new ST(Constants.EVS_Language_code), new ST(Constants.MIME_TEXT), null);
                        itemViewModel.Model.Properties.Add(property);
                        itemViewModel.ReasonType = input.UnavailableReasonTypeList.SingleOrDefault(cp => cp.code == input.UnavailableReasonType).displayName;
                        itemViewModel.LoggedUser = this.loggedUser;
                        FacadeManager.VocabularyInterface.InsertConcept(itemViewModel.Model);
                        return View("AddConfirmation", itemViewModel);
                    }
                    else
                        return View("UnavailableReasonAlreadyExists", itemViewModel);
                }
                else
                    return View("MissingUnavailableReasonText", input);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Action for redirecting to the Edit view.
        /// </summary>
        /// <param name="id">The id of the unavailable reason object selected.</param>
        /// <returns>The Edit view.</returns>
        [ReadPermissionAuthFilterAttribute]
        public ActionResult Edit(string p)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                UnavailableReasonViewModel viewModel = new UnavailableReasonViewModel();
                FillViewModel(viewModel);
                UnavailableReasonItemViewModel itemViewModel = viewModel.UnavailableReasonListExtended.SingleOrDefault(a => a.Model.code == QueryStrings["code"]);
                viewModel.UnavailableReasonCode = itemViewModel.Model.code;
                viewModel.UnavailableReasonText = itemViewModel.Model.displayName;
                viewModel.UnavailableReasonType = itemViewModel.Model.Properties.SingleOrDefault(a => a.PropertyName.text == "Type").PropertyValue.text;
                viewModel.UnavailableReasonTypeText = itemViewModel.ReasonType;
                return View(viewModel);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Post action for the Edit view - used for the Submit button.
        /// </summary>
        /// <param name="input">The input(unavailable reason view model).</param>
        /// <returns>The Edit Confirmation view.</returns>
        [HttpPost]
        [UpdatePermissionAuthFilterAttribute]
        public ActionResult Edit(UnavailableReasonViewModel input)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                ModelState.Clear();
                if (string.IsNullOrWhiteSpace(input.UnavailableReasonText))
                    return View("MissingUnavailableReasonText", input);
                List<CDWithProperties> reasons = FacadeManager.VocabularyInterface.GetAllUnavailableReasons();
                CDWithProperties concept = reasons.SingleOrDefault(a => a.code.Equals(input.UnavailableReasonCode, StringComparison.InvariantCultureIgnoreCase));
                input.Message = Strings.EditConfirmationText;
                input.DisplayButtonWhenMissingReasonText = Strings.DisplayNone;
                input.DisplayButtonWhenSuccessfullyEditReasonText = Strings.DisplayBlock;
                if (concept != null)
                {
                    concept.displayName = input.UnavailableReasonText.ToUpper().Trim();
                    //check if another CD with the same display name already exists
                    if (reasons.Where(a => a.displayName.Equals(concept.displayName, StringComparison.InvariantCultureIgnoreCase) &&
                            !a.code.Equals(concept.code, StringComparison.InvariantCultureIgnoreCase)).FirstOrDefault() != null)
                    {
                        input.Message = Strings.UnavailableReasonAlreadyExists;
                        input.DisplayButtonWhenMissingReasonText = Strings.DisplayBlock;
                        input.DisplayButtonWhenSuccessfullyEditReasonText = Strings.DisplayNone;
                    }
                    else
                    {
                        ConceptProperty property = concept.Properties.SingleOrDefault(a => a.PropertyName.text == "Type");
                        property.PropertyValue.text = input.UnavailableReasonType;
                        concept.Properties.Clear();
                        concept.Properties.Add(property);
                        FacadeManager.VocabularyInterface.UpdateConcept(concept);
                    }
                }
                return View("EditConfirmation", input);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Action for redirecting to the Delete view.
        /// </summary>
        /// <param name="code">The id of the unavailable reason object selected.</param>
        /// <returns>The Delete view.</returns>
        [ReadPermissionAuthFilterAttribute]
        public ActionResult Delete(string p)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                UnavailableReasonViewModel viewModel = new UnavailableReasonViewModel();
                FillViewModel(viewModel);
                UnavailableReasonItemViewModel itemViewModel = viewModel.UnavailableReasonListExtended.SingleOrDefault(a => a.Model.code == QueryStrings["code"]);
                viewModel.UnavailableReasonCode = itemViewModel.Model.code;
                viewModel.UnavailableReasonText = itemViewModel.Model.displayName;
                viewModel.UnavailableReasonType = itemViewModel.Model.Properties.SingleOrDefault(a => a.PropertyName.text == "Type").PropertyValue.text;
                viewModel.UnavailableReasonTypeText = itemViewModel.ReasonType;

                return View(viewModel);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Post action for the Delete view - used for the Delete button.
        /// </summary>
        /// <param name="input">The input(unavailable reason view model).</param>
        /// <returns>The Delete Confirmation view.</returns>
        [HttpPost]
        [UpdatePermissionAuthFilterAttribute]
        public ActionResult Delete(UnavailableReasonViewModel input)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                ModelState.Clear();
                BaseViewModel baseViewModel = new BaseViewModel();
                baseViewModel.LoggedUser = this.loggedUser;

                UnavailableReasonViewModel viewModel = new UnavailableReasonViewModel();
                FillViewModel(viewModel);
                UnavailableReasonItemViewModel itemViewModel = viewModel.UnavailableReasonListExtended.SingleOrDefault(a => a.Model.code == input.UnavailableReasonCode);

                if (itemViewModel != null)
                    FacadeManager.VocabularyInterface.DeleteConcept(itemViewModel.Model);

                return View("DeleteConfirmation", baseViewModel);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        /// <summary>
        /// Fills the view model properties with data from Facade.
        /// </summary>
        /// <param name="hm">The unavailable reason model.</param>
        private void FillViewModel(UnavailableReasonViewModel viewModel)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                viewModel.UnavailableReasonTypeList = FacadeManager.VocabularyInterface.GetVocabulary(Facade.Data.Util.Vocabulary.AdminURType);
                viewModel.UnavailableReasonList = FacadeManager.VocabularyInterface.GetReasons().Where(a => a.code.StartsWith(Constants.NATIONAL_VOCABULARY_CODE_PREFIX)).ToList();
                viewModel.LoggedUser = this.loggedUser;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

    }
}
